<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\ActivationCode;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;

class UserAuthController extends Controller
{
    /**
     * Login user and return Sanctum token
     */
    public function login(Request $request)
    {
        $request->validate([
            'email'    => 'required|email',
            'password' => 'required',
        ]);

        $user = User::where('email', $request->email)->first();

        if (! $user || ! Hash::check($request->password, $user->password)) {
            throw ValidationException::withMessages([
                'email' => ['بيانات الدخول غير صحيحة'],
            ]);
        }

        // Revoke old tokens and issue new one
        $user->tokens()->delete();
        $token = $user->createToken('user-panel')->plainTextToken;

        return response()->json([
            'token' => $token,
            'user'  => [
                'id'    => $user->id,
                'name'  => $user->name,
                'email' => $user->email,
            ],
        ]);
    }

    /**
     * Register a new user (after activation)
     */
    public function register(Request $request)
    {
        $request->validate([
            'name'     => 'required|string|max:100',
            'email'    => 'required|email|unique:users',
            'password' => 'required|min:8|confirmed',
        ]);

        $user = User::create([
            'name'     => $request->name,
            'email'    => $request->email,
            'password' => Hash::make($request->password),
        ]);

        $token = $user->createToken('user-panel')->plainTextToken;

        return response()->json([
            'token' => $token,
            'user'  => [
                'id'    => $user->id,
                'name'  => $user->name,
                'email' => $user->email,
            ],
        ], 201);
    }

    /**
     * Logout / revoke tokens
     */
    public function logout(Request $request)
    {
        $request->user()->tokens()->delete();
        return response()->json(['message' => 'تم تسجيل الخروج']);
    }

    /**
     * Get authenticated user profile + active subscription
     */
    public function profile(Request $request)
    {
        $user = $request->user();
        $code = ActivationCode::where('user_id', $user->id)
            ->where('status', 'active')
            ->latest()
            ->first();

        return response()->json([
            'user' => [
                'id'    => $user->id,
                'name'  => $user->name,
                'email' => $user->email,
            ],
            'subscription' => $code ? [
                'code'       => $code->code,
                'type'       => $code->type,
                'status'     => $code->status,
                'expires_at' => $code->expires_at?->toDateTimeString(),
                'days_left'  => $code->expires_at ? now()->diffInDays($code->expires_at, false) : null,
                'device_count' => $code->device_count,
            ] : null,
        ]);
    }

    /**
     * Update user profile details
     */
    public function updateProfile(Request $request)
    {
        $user = $request->user();

        $request->validate([
            'name'  => 'sometimes|string|max:100',
            'email' => 'sometimes|email|unique:users,email,' . $user->id,
            'phone' => 'sometimes|nullable|string|max:20',
            'city'  => 'sometimes|nullable|string|max:100',
        ]);

        $user->update($request->only(['name', 'email', 'phone', 'city']));

        return response()->json(['message' => 'تم تحديث البيانات', 'user' => $user]);
    }

    /**
     * Change user password
     */
    public function changePassword(Request $request)
    {
        $request->validate([
            'current_password' => 'required',
            'password'         => 'required|min:8|confirmed',
        ]);

        $user = $request->user();

        if (! Hash::check($request->current_password, $user->password)) {
            return response()->json(['message' => 'كلمة المرور الحالية غير صحيحة'], 422);
        }

        $user->update(['password' => Hash::make($request->password)]);

        return response()->json(['message' => 'تم تغيير كلمة المرور بنجاح']);
    }

    /**
     * Get all subscriptions history for this user
     */
    public function subscriptions(Request $request)
    {
        $codes = ActivationCode::where('user_id', $request->user()->id)
            ->latest()
            ->get()
            ->map(fn($c) => [
                'id'         => $c->id,
                'code'       => $c->code,
                'type'       => $c->type,
                'status'     => $c->status,
                'expires_at' => $c->expires_at?->toDateString(),
                'created_at' => $c->created_at->toDateString(),
                'days_left'  => $c->expires_at ? max(0, now()->diffInDays($c->expires_at, false)) : null,
            ]);

        return response()->json($codes);
    }
}
